/* ==================================================================
File        : deferredFilterLighting.hlsl
Author      : Vassilis Poulopoulos
Date        : 15/07/2008 11:19:14
Format      : HLSL/Cg
Description : Implementation of deferred lighting

================================================================== */

#include "DeferredFilterShadows.hlsl"

//------------------------------------------------------------------------------------------------
// Shader constants
//------------------------------------------------------------------------------------------------

struct LightData
{
	float4		posDirRadius;	// XYZ = position/direction in eye space						W = -1/(outer-inner radius)
	float4		color;			// RGB = baseColor												A = outer radius / (outer-inner radius)
	float4		params1;		// X = specular factor	Y = ambient factor	Z = diffuse factor	W = <spare>
};

LightData	g_Light			: register(c180);
rm float4x4	g_SpotViewProj	: register(c183);	// spot light ViewProj matrix
float4		g_LightCommon	: register(c187);	// .x = specular intensity range, .yzw = <unused>

// Used for 2nd forward light
LightData	g_Light1		: register(c190);
rm float4x4	g_SpotViewProj1	: register(c193);	// spot light ViewProj matrix

//------------------------------------------------------------------------------------------------
// Calculates the lighting at a point in space
//------------------------------------------------------------------------------------------------
void ComputeLighting(	half3 L, float3 P, half3 N, half3 N_Specular, half shadowFactor, 
						LightData light,
						half materialSpecularPower, half3 materialSpecularIntensity, half materialHalfLambertFactor,
						out half3 outDiffuseLight,
						out half3 outSpecularLight)
{
	half3 I = normalize(-P);
	half3 H = normalize(I + L);
	half3 R = -reflect(L, N_Specular);

	half specularFactor = light.params1.x;
	half ambientFactor = light.params1.y;
	half diffuseFactor = light.params1.z;

	half diffuseDotFull = dot(N, L);
	half diffuseDotClamp = saturate(diffuseDotFull);

	half diffuseDot = lerp(diffuseDotClamp, diffuseDotFull * 0.5 + 0.5, materialHalfLambertFactor);
	//half specularDot = saturate(dot(N_Specular, H));
	float specularDot = saturate(dot(I, R));

	// Have to do this to avoid pow(0,0) below. The specular power is sometimes zero when no deferred lights
	// do specular (i.e. specularFactor is zero), in which case we skip the step that fills in the specular power
	// in the stencilValues texture.
	materialSpecularPower += 0.001; 
	outSpecularLight = specularFactor * materialSpecularIntensity * pow(specularDot, materialSpecularPower) * shadowFactor;

#if defined(UBERSHADER) && (LIGHTING_MODEL == LIGHTING_SKIN)
//	shadowFactor = lerp( shadowFactor, 1 - (1-shadowFactor) * diffuseDotClamp, materialHalfLambertFactor );

	half diffuseDotUnShadowed = lerp(diffuseDotClamp, diffuseDotFull * 0.5 + 0.5, materialHalfLambertFactor);

	half diffuseDotShadowed = lerp(0, -abs(diffuseDotFull) * 0.5 + 0.5, materialHalfLambertFactor);

	diffuseDot = lerp(diffuseDotShadowed, diffuseDotUnShadowed, shadowFactor);

	shadowFactor = 1;
#endif

	outDiffuseLight  = diffuseFactor * diffuseDot * shadowFactor + ambientFactor;
	
#if defined(NO_SPECULAR)
	outSpecularLight = 0;
#endif

#if defined(NO_DIFFUSE)
	outDiffuseLight = ambientFactor;
#endif

	outDiffuseLight		*= light.color.rgb;
	outSpecularLight	*= light.color.rgb;
}

#if defined(UBERSHADER)

//------------------------------------------------------------------------------------------------
// Calculates the lighting at a point in space using the Anisotropic lighting model
//------------------------------------------------------------------------------------------------
void ComputeLightingAnisotropic(	half3 L, float3 P, half3 N, half3 N_Specular, half shadowFactor, 
									LightData light,
									AnisotropicSurfaceProperties anisoSurfaceProperties,
									half materialSpecularPower, half3 materialSpecularIntensity, half materialHalfLambertFactor,
									out half3 outDiffuseLight,
									out half3 outSpecularLight)
{
	half3 I = normalize(-P);
	half3 H = normalize(I + L);
	half3 R = -reflect(L, N_Specular);

	half specularFactor = light.params1.x;
	half ambientFactor = light.params1.y;
	half diffuseFactor = light.params1.z;


	half ldotb	= dot(L, anisoSurfaceProperties.anisoBasis);
	half ad		= sqrt(1.0 - ldotb * ldotb);

	half ldotbs	= dot(L, anisoSurfaceProperties.anisoBasisS);
	half ads	= sqrt(1.0 - ldotbs * ldotbs);

	// Diffuse
	outDiffuseLight = diffuseFactor * saturate(dot(N, L)) * shadowFactor * ad + ambientFactor;

	// Specular
	half ldotns			= saturate(dot(L, N_Specular));
	half base			= saturate(ads * anisoSurfaceProperties.vbf - (ldotbs * anisoSurfaceProperties.vdotb));
	outSpecularLight	= materialSpecularIntensity * specularFactor * ldotns * pow(base, materialSpecularPower) * shadowFactor;

#if defined(NO_SPECULAR)
	outSpecularLight = 0;
#endif

#if defined(NO_DIFFUSE)
	outDiffuseLight = ambientFactor;
#endif

	outDiffuseLight		*= light.color.rgb;
	outSpecularLight	*= light.color.rgb;
}
#endif


#if !defined(UBERSHADER)

//------------------------------------------------------------------------------------------------
// Main deferred lighting pixel shader. 
//------------------------------------------------------------------------------------------------
float4 DeferredLightPS(VS_OUT interpolants) : COLOR
{
	float4 coords = interpolants.coords;

	#if !defined(DIR_LIGHT)
	// We have saved the w component of the fragment position into interpolants.projectionParams.y
	coords /= interpolants.projectionParams.y;
	#endif

	// fragment position (in eye space)
	float4 P = BackProject(coords.xy, coords.zw, interpolants.projectionParams);


	//------------------------------------------------
	// Calculate shadow intensity
	//------------------------------------------------

	half shadowIntensity = 0;

#if defined(CALC_SHADOW)

	// Calculate fragment position in light space
	SoftShadowParams softShadowParams;
	float4 fragPosShadowSpace = ProjectToShadowMap(P, softShadowParams);

	// Compare fragment against the shadowmap
	shadowIntensity = LookupShadowmap(fragPosShadowSpace, P, softShadowParams);

	// Check if we shouldn't apply shadows to this pixel
	float4 stencilValues = tex2D(g_StencilValuesTex, coords.xy);
	shadowIntensity *= 1.0f - stencilValues.r;

#elif defined(LUMINANCE_MAP_SHADOW)

	shadowIntensity = tex2D(g_LuminanceTex, coords.xy).r;

#endif

	//------------------------------------------------
	// Light geometry
	//------------------------------------------------

	// light direction (in eye space)
	#if defined(DIR_LIGHT)
	
	half3 L = -g_Light.posDirRadius;
	half attenuationDistance = 0;

	#else
	half3 L = g_Light.posDirRadius.xyz - P.xyz;

	// Normalise light direction
	half attenuationDistance = length(L);

	L = normalize(L); // same instruction count as L /= distance;
	#endif

	//------------------------------------------------
	// Calculate attenuation
	//------------------------------------------------

	half attenuationFactor = 1.0;

	// Attenuate with distance from light
	#if !defined(DIR_LIGHT)
	attenuationFactor *= saturate(attenuationDistance * g_Light.posDirRadius.w + g_Light.color.a);
	#endif

	half4 lightTexture = 1;

	// Spot light attenuation (radial distance away from light direction vector)
	#if defined(SPOT_LIGHT)
	// Calculate fragment position in spot light space
	half4 fragPosLight = mul(P, g_SpotViewProj);
	fragPosLight.xyz /= fragPosLight.w;

		#if !defined(SPOT_TID)

			// Radial attenuation
			half radialDistanceSq = fragPosLight.x * fragPosLight.x + fragPosLight.y * fragPosLight.y;
			attenuationFactor *= saturate(1 - radialDistanceSq);

		#else

			lightTexture = tex2D(g_SpotTex, fragPosLight.xy * float2(0.5, -0.5) + 0.5);
			attenuationFactor *= lightTexture.a;

		#endif

	#endif

	//------------------------------------------------
	// Calculate lighting
	//------------------------------------------------
	if (shadowIntensity < 0.99)
	{
		half4 albedo		= tex2D(g_AlbedoTex, coords.xy);
		half4 stencilValues = tex2D(g_StencilValuesTex, coords.xy);
		
		half4 normalTex		= tex2D(g_NormalsTex, coords.xy);

		half specularPower = stencilValues.a * 100.0;
		half3 specularIntensity = normalTex.b * g_LightCommon.x;

		half3 N = normalTex.xyz * 2.0 - 1.0;
		N.z = -sqrt(1 - N.x*N.x - N.y*N.y);

		//half3 N;
		//float2 fenc = normalTex.xy*2-1;
		//N.z = -(dot(fenc,fenc)*2-1);
		//N.xy = normalize(fenc) * sqrt(1-N.z*N.z);

		//float scale = 1.7777;
		//float3 nn = normalTex.xyz*float3(2*scale,2*scale,0) + float3(-scale,-scale,1);
		//float g = 2.0 / dot(nn.xyz,nn.xyz);
		//N.xy = g*nn.xy;
		//N.z = g-1;


		half3 diffuseLight = 0;
		half3 specularLight = 0;
		ComputeLighting(	L, P.xyz, N, N, saturate(1.0 - shadowIntensity), 
							g_Light,
							specularPower, specularIntensity, 0,
							diffuseLight, specularLight);

		float3 surfaceColor = diffuseLight * albedo + specularLight;

		surfaceColor.rgb *= lightTexture.rgb;

		return float4(surfaceColor.rgb, attenuationFactor);
	}
	else
	{
		float4 albedo = tex2D(g_AlbedoTex, coords.xy);
		float ambientFactor = g_Light.params1.y;

		return float4(albedo * g_Light.color.rgb * lightTexture.rgb * ambientFactor, attenuationFactor);
	}
}

#endif // UBERSHADER
